//
//  Trigonometry.swift
//  Do It
//
//  Created by Jim Dovey on 2/5/20.
//  Copyright © 2020 Jim Dovey. All rights reserved.
//

import Foundation
import SwiftUI

extension CGPoint {
    static func + (lhs: CGPoint, rhs: CGPoint) -> CGPoint {
        CGPoint(x: lhs.x + rhs.x, y: lhs.y + rhs.y)
    }
    static func + (lhs: CGPoint, rhs: CGSize) -> CGPoint {
        lhs + CGPoint(x: rhs.width, y: rhs.height)
    }
    static func + (lhs: CGPoint, rhs: CGFloat) -> CGPoint {
        CGPoint(x: lhs.x + rhs, y: lhs.y + rhs)
    }

    static func - (lhs: CGPoint, rhs: CGPoint) -> CGPoint {
        CGPoint(x: lhs.x - rhs.x, y: lhs.y - rhs.y)
    }
    static func - (lhs: CGPoint, rhs: CGSize) -> CGPoint {
        CGPoint(x: lhs.x - rhs.width, y: lhs.y - rhs.height)
    }
    static func - (lhs: CGPoint, rhs: CGFloat) -> CGPoint {
        CGPoint(x: lhs.x - rhs, y: lhs.y - rhs)
    }

    func offset(by distance: CGFloat, at angle: CGFloat) -> CGSize {
        CGSize(width: cos(angle) * distance,
               height: sin(angle) * distance)
    }

    func distance(from point: CGPoint) -> CGFloat {
        (self - point).distance()
    }
    func distance() -> CGFloat {
        hypot(x, y)
    }
    func angle(from point: CGPoint) -> CGFloat {
        (self - point).angle()
    }
    func angle() -> CGFloat {
        distanceAndAngle().angle
    }

    func distanceAndAngle(from point: CGPoint) -> (distance: CGFloat, angle: CGFloat) {
        (self - point).distanceAndAngle()
    }
    func distanceAndAngle() -> (distance: CGFloat, angle: CGFloat) {
        let radius = distance()
        if radius == 0 {
            return (radius, 0)
        }
        else {
            let angle = acos(x / radius) / .pi / 2
            return (radius, angle)
        }
    }

    func boundedInCircle(radius: CGFloat) -> CGPoint {
        var loc = self
        let offset = self - radius
        if offset.distance() > radius {
            let theta = atan2(offset.y, offset.x)
            loc.x = radius * cos(theta) + radius
            loc.y = radius * sin(theta) + radius
        }
        return loc
    }

    func unit(within bounds: CGSize) -> CGPoint {
        CGPoint(x: x / (bounds.width/2), y: y / (bounds.height/2))
    }
    func centeredUnit(within bounds: CGSize) -> CGPoint {
        unit(within: bounds) - 1.0
    }

    mutating func bound(in radius: CGFloat) {
        self = boundedInCircle(radius: radius)
    }
}
